;******************************************************************************
; 	This firmware is offered as is with no expectations of assistance from
;	Intersil.
; 
;	For technical support for the PIC18F2431, please contact Microchip.
;******************************************************************************
	include		"P18F2431.inc"
	include 	"HIP4086 Motor Drive Parameters(ver 2).inc"
;******************************************************************************
#define	VersionNumber	0x02	;This is the version number of this code.
;..............................................................................
; WARNING:
;	1. no RAM bank management is used in this code example because of its small
;	   size. Max RAM address is < 0x100. Only Access RAM is used.
;	2. No ROM addressing management is used because of this example code's 
;      small size. Max PC value is << 0x1000.
;..............................................................................
STARTUP		code 0x00
	bra	START_
INT_HIGH 	code 0x08
	bra	ISR_HIGH
INT_LOW 	code 0x018
	bra	ISR_LOW
	nop


HIP4086		code 0x0020
START_
;..............................................................................
; PORTA is initialized later when the ADC is configured
;..............................................................................
;..............................................................................
; Initialize PORTB
; RB0 .. RB5 are used to drive the inputs of the HIP4086. 
; Turn on all low side FETs and turn off all high side FETs. Note that RB1,RB3,
; and RB4 (which correspond to /AHI, /BHI, and /CHI on the HIP4086) must be set
; high to turn off the high FETs.
;
; Note that the start-up states of PORTB are setup in the hardware configuration
; bytes of the microcontroller.
;..............................................................................
	clrf	PORTB		
	movlw	TrisbValue
	movwf	TRISB
;..............................................................................
; Initialize PORTC
; PortC interfaces to push-buttons, dip switches and LEDs. Initially, This port
; is configured to drive LEDs (the default mode).  The LEDS are off at this
; point and are turned on, with various combinations, elsewhere in the program. 
;
; RC4, RC5, RC6, and RC7 are muxed i/o pins. When RC0 is set, the push-buttons
; are read, when RC2 is set, the dip switches are read, and when RC3 is cleared
; the LEDs are active. (for more information, see the EnableMux macro) 
;..............................................................................
	clrf	LATC		
	movlw	LEDmux			
	movwf	TRISC
;..............................................................................
; clear all ram after reset, then initialize variables
;..............................................................................	
	call 	CLEAR_RAM				
	bcf		StopMotor_b			;setup to start the motor
	movlw	DebounceCnt
	movwf	DebounceCntr
;..............................................................................
; Initialize ADC,PORTA,B,C, PWM controller etc.
;..............................................................................	
	call	INIT_PERIPHERALS	
;..............................................................................
; Special contitions must be met prior to turn-on to initial the test mode.
; For specific requirements, see TEST_MODE
;..............................................................................
	Call	TEST_MODE		
;..............................................................................
; Initialize Required Interrupts
;..............................................................................
	movlw	0x093				;Power ON reset status bit/Brownout reset status bit
	movwf	RCON				;and Instruction flag bits are set
	bcf		HallAintFlag_b
	bcf		HallBintFlag_b
	bcf		HallCintFlag_b
	bsf		ADCintEnable_b		;AD Converter over Interrupt enable
	bsf		HallAintEnable_b	;Input Capture1 interrupt enabled
	bsf		HallBintEnable_b	;Input Capture2 interrupt enabled
	bsf		HallCintEnable_b	;Input Capture3 interrupt enabled
	bsf		PWMintEnable_b		;PWM interrupt enabled
	bsf		Timer0IntEnable_b	;turn on timer 0 for blinking LED
	bsf		Timer1IntEnable_b	;turn on for motor not rotating detection
	bsf		PeripheralIntEn_b	;Peripheral interrupts enable
	bsf		GlobalIntEnable_b	;Global interrupt enable
;..............................................................................
; The dip switch must be setup for the desired bldc switching sequence prior to
; applying power or before pushing the reset button.
;..............................................................................
	
	Call 	RD_DIP_SWITCH
	TurnOnLED LED0,NoBlink
	movlw	10					
	Call	DELAY_
	TurnOffLED LED0			;Flash LEDs to indicate power has been applied
	TurnOnLED LED1,NoBlink
	movlw	10					
	Call	DELAY_
	TurnOffLED LED1			;Flash LEDs to indicate power has been applied
	TurnOnLED LED2,NoBlink
	movlw	10					
	Call	DELAY_
	TurnOffLED LED2			;Flash LEDs to indicate power has been applied
	TurnOnLED LED3,NoBlink
	movlw	10					
	Call	DELAY_
	TurnOffLED LED3			;Flash LEDs to indicate power has been applied
	
	bsf		ForwardMotor_b
	bsf		StopMotor_b
	bcf	 	SeqEnabled_b
	call 	UPDATE_SEQUENCE		;This is the initial setup for the bridge FETs									
;******************************************************************************
;Control keeps revolving in the main loop
;PWM duty cycle is updated after every ADC conversion
;When a push button switch is pressed, action is taken based on the switch pressed
;******************************************************************************
MainLoop 
	btfss	 ADCenConvert_b	
		bsf	 ADCenConvert_b		;Start the ADC conversion.
 	btfsc	 UpdatePWM_b		;When set, ADC has provided new values
		call UPDATE_PWM			;Update PWM duty cycle
	call	 PBUTTON_STATUS		;get status of Start/Stop, Reverse, Brake switches
	call	 PROCESS_PBUTTONS	;Service switches
	bra	 	 MainLoop
;..............................................................................
 	
ISR_LOW
;******************************************************************************
;Low priority ISR  not used in this code example
;******************************************************************************
	bra 	freeze1				;used for troubleshooting
	RETFIE	FAST		
freeze1
	goto $
	
ISR_HIGH 
;******************************************************************************
; High priority Interrupt service
; ADC, Hall sensor status, and PWM ISRs are serviced here.
;******************************************************************************
	movff	STATUS,STATUSsave
ChkInterrupts
	btfsc	HallAintFlag_b		;Hall A sensor interrupt
		bra HALL_A_INT
	btfsc	HallBintFlag_b		;Hall B sensor interrupt
		bra HALL_B_INT
	btfsc	HallCintFlag_b		;Hall C sensor interrupt
		bra HALL_C_INT
	btfsc	ADCintFlag_b		;ADC interrupt
		bra ADC_INT	
	btfsc	PWMintFlag_b		;PWM interrupt
		bra PWM_INTERRUPT
	btfsc	Timer0IntFlag_b
		bra	TIMER0_INT
	btfsc	Timer1IntFlag_b
		bra	TIMER1_INT
	bra		xISR_HIGH			;this shouldn't happen
;..............................................................................
;Timer 0 Interrupt blinks the LEDs
;   This routine controls that blinking state of the four LEDs. The bits in 
;	EnableLED_B are set elsewhere to turn on the LED as specified by bits b4..
;	..b7. The bits in EnableBlink_B are set elsewhere to cause an LED to blink
;	on and off. Bits 4,5,6, and 7 also correspond to the same bits of PORTC 
;	that drive the LEDs. 
;
;	The logic is somewhat difficult to follow. The following example illustrates
;	the logic with the b4 LED blinking and the b5 LED on without blinking:
;	
;	initial values:
;   ToggleBlink_b = 1
;	EnableBlink_B = 00010000	(b4 is set to blink the b4 LED)
;	 EnableLEDs_B = 00110000	(b4 and b5 are set to turn on LEDs on b4 and b5)
;	 						  
;	movf	EnableBlink_B,W : 00010000 (the value of EnableBlink_b is in Wreg)
;	iorwf	EnableLEDs_B,W  : 00110000 	
;	movwf	LATC  			: 00110000 (b4 and b5 are on)
;
;	initial values:
;   ToggleBlink_b = 0
;	EnableBlink_B = 00010000	(b4 is set to blink the b4 LED)
;	 EnableLEDs_B = 00110000	(b4 and b5 are set to turn on LEDs on b4 and b5)
;	 						  
;	movlw	0x0F			: 00001111	
;	iorwf	EnableBlink_B,W : 00011111
;	 comf	W,W			  	: 11100000
;	andwf	EnableLEDs_B,W	: 00100000
;	movwf	LATC			: 00100000 (b4 is blinked off, b5 is still on)
;..............................................................................
TIMER0_INT
	btg		ToggleBlink_b
	btfss	ToggleBlink_b
		bra	BlinkOff
BlinkOn
	movf	EnableBlink_B,W
	iorwf	EnableLEDs_B,W
	bra		xTIMER0_INT
BlinkOff
	movlw	0x0F
	iorwf	EnableBlink_B,W
	comf	WREG
	andwf	EnableLEDs_B,W
xTIMER0_INT
	movwf	LATC
	bcf		Timer0IntFlag_b
	bra 	ChkInterrupts
	
;..............................................................................
;When Timer1 overflows (every 6.554 msec), the RotationCntr is decremented.
;When RotationCntr = 0, the motor has stopped rotating and a flag is set.
;While the motor is rotating, the HALL_A_INT interrupt sets RotationCntr = 38
;for a delay of 250 msec. 
;..............................................................................
TIMER1_INT
	bcf		Timer1IntFlag_b
	dcfsnz	RotationCntr
		bcf	MotorRunning_b
	bra		ChkInterrupts
;..............................................................................
; Hall sensor ISR on IC1,IC2 and IC3 (a.k.a. PA2, PA3, and PA4)
; Update the winding energizing sequence.
;..............................................................................
HALL_A_INT 
	movlw	38
	movwf	RotationCntr	 
	bsf		MotorRunning_b
	bcf		HallAintFlag_b
	call	UPDATE_SEQUENCE		;Update the winding energizing sequence.
	btfss	TestModeActive_b
		bra ChkInterrupts
	TurnOnLED LED0,NoBlink
	bra		ChkInterrupts
HALL_B_INT 
	bcf		HallBintFlag_b
	call	UPDATE_SEQUENCE		;Update the winding energizing sequence.
	btfss	TestModeActive_b
		bra xISR_HIGH
	TurnOnLED LED1,NoBlink
	bra		ChkInterrupts
HALL_C_INT 
	bcf		HallCintFlag_b
	call	UPDATE_SEQUENCE		;Update the winding energizing sequence.
	btfss	TestModeActive_b
		bra xISR_HIGH
	TurnOnLED LED2,NoBlink
	bra		ChkInterrupts
;..............................................................................
; ADC ISR
; The motor current and the speed setpoint is obtained from the ADC
;..............................................................................
ADC_INT	
	btfsc	ADCbufOverFlow_b
		bra freeze					;just used for troubleshooting
	movff	ADRESH,MotorCurrent_H	;Sample A = Motor current
	movff	ADRESL,MotorCurrent_L	;(10 bit result is right shifted)
	movff	ADRESH,SpeedRef_H		;Sample B = speed ref
	movff	ADRESL,SpeedRef_L		;(10 bit result is right shifted)
	bsf		UpdatePWM_b				;Set flag to indicate ADC conversion complete
	bcf		ADCintFlag_b			;ADIF flag is cleared for next interrupt
	bra		ChkInterrupts
freeze
	bra $							;stop here if an ADC buffer over flow
									; has occured.
		
;..............................................................................
; This ISR checks for over current Faults. 
; Over current (FLTA) is initialized in cycle-by-cycle mode. 
;
; If OC faults occur very frequently, the mode is changed to 
; catastrophic mode and PWM is stopped. The occurence of  an OC
; fault is checked every PWM interrupt with a limit(MaxFaultAcnt)
; and if it exeeds the limit in 256 PWM cycles, the mode is changed
; to catastrophic.
;..............................................................................
PWM_INTERRUPT	
	incfsz	PWMcycleCntr,F
		bra	ChkForFaults
	clrf	FaultAcntr
	bra	tstLEDstatus
ChkForFaults
	btfss	FLTCONFIG,FLTAS
		bra	tstLEDstatus
	incf	FaultAcntr,F
	movlw	MaxFaultAcnt
	cpfsgt	FaultAcntr
		bra	tstLEDstatus	
	clrf	OVDCOND
	movlw	b'01000111'
	movwf	PWMCON0		 		;PWM output pairs are in independant mode
	bcf		FLTCONFIG,FLTAMOD
	bsf		OverCurFault_b
tstLEDstatus
	btfss	OverCurFault_b
		bra xPWM_INTERRUPT
	TurnOnLED Ilimit,Blink
	TurnOffLED Run
xPWM_INTERRUPT
	bcf		PWMintFlag_b

xISR_HIGH
	movff	STATUSsave,STATUS
	RETFIE	FAST
		
UPDATE_PWM 
;*******************************************************************************
; This routine updates the PWM duty cycle according to the input speed
; referance as measured by the ADC on the AN0 input. The required PWM is simply  
; the value from AN0 scaled (1/2) for the maximum PTPER (time base period).
; Limits are also applied for the minimum and maximum duty cycle (empherically
; derived). 
;
; Because the actual motor to be used on this eval board is not specified,
; the rate of change of PWM is also limited to minimize application issues.
;*******************************************************************************
	bcf		UpdatePWM_b				;This bit is set by ADC_INT
	btfsc	StopMotor_b
		bra xUPDATE_PWM				;update PWM only if motor is running
	decfsz	UpdateDelayCntr
		bra	xUPDATE_PWM
UpdateCntr
	movlw	UpdateDelayCnt
	movwf	UpdateDelayCntr
;..............................................................................
; Divide SpeedRef by two and save in PWMsetting to scale the duty cycle value
; with the PTPER (time base) value. Note for 20KHz, PTPER = 0x1F3 and the max
; value for SpeedRef = 0x3ff (10 bit ADC). 
;..............................................................................
	movff	SpeedRef_L,PWMsetting_L
	movff	SpeedRef_H,PWMsetting_H
	bcf		STATUS,C				;clear carry bit
	rrcf	PWMsetting_H
	rrcf	PWMsetting_L			;max value for PWMsetting = 0x1ff
;..............................................................................
;Set a minimum or max limit to PWMsetting
;..............................................................................

; 16-bit Subtraction, (PWMsetting - MinPWM)
TestMinLimit
        movlw	high(MinPWM)
        movwf	LimitPWM_H
        movlw  	low(MinPWM)
        movwf	LimitPWM_L    		;LimitPWM = MinPWM
        subwf   PWMsetting_L,W		;W = PWMsetting_L - LimitPWM_L
        movf    LimitPWM_H,W		
        subwfb  PWMsetting_H,W		;W = PWMsetting_H - LimitPWM_H
		bnn		TestMaxLimit
		bra		SetPWMlimit

; 16-bit Subtraction, (PWMsetting - MaxPWM)
TestMaxLimit
        movlw	high(MaxPWM)
        movwf	LimitPWM_H
        movlw  	low(MaxPWM)
        movwf	LimitPWM_L    		;LimitPWM = MaxPWM
        subwf   PWMsetting_L,W		;W = PWMsetting_L - LimitPWM_L
        movf    LimitPWM_H,W		
        subwfb  PWMsetting_H,W		;W = PWMsetting_H - LimitPWM_H
		bn		RateLimit
SetPWMlimit
		movff	LimitPWM_L,PWMsetting_L
		movff	LimitPWM_H,PWMsetting_H
				
;....... 16-bit Subtraction example (R = A - B)................................
;        movf    B0,W
;        subwf   A0,W
;        movwf   R0
;        movf    B1,W
;        subwfb  A1,W
;        movwf   R1
;..............................................................................
;..............................................................................
;The rate of change of PWM is limited
;..............................................................................	
RateLimit
	movf	PWMsetting_H,w
	subwf	PWMdutyCycle_H,w	; w = PWMdutyCycle_H - PWMsetting_H
	bz		ChkLowerByte
	bn		IncrPWMdc
	bra		DecrPWMdc
ChkLowerByte
	movf	PWMsetting_L,w
	subwfb	PWMdutyCycle_L,w	; w = PWMdutyCycle_L - PWMsetting_L
	bz		Nochange			; no change, PWMsetting = PWMdutyCycle
	bnc		IncrPWMdc
DecrPWMdc
	decf	PWMdutyCycle_L
	bc		UpdatePDCs
	decf	PWMdutyCycle_H
	bra		UpdatePDCs	
IncrPWMdc
	incf	PWMdutyCycle_L
	bnz		UpdatePDCs
	incf	PWMdutyCycle_H
;..............................................................................
;Load PDCxH/L registers with the duty cycle value in PWMdutyCycle
;..............................................................................	
UpdatePDCs
	movff	PWMdutyCycle_H, PDCbuf_H
	movff	PWMdutyCycle_L, PDCbuf_L
	call	LOAD_PDC
xUPDATE_PWM
	return
Nochange
	nop
	return							;this is used just for troubleshooting
	
;******************************************************************************
; Update the PDC registers with values in PDCH and PDCL.
;******************************************************************************
LOAD_PDC	
	bsf		PWMCON1,UDIS		;Disable the PWM buffer update
	bcf		STATUS,C			;shift PDC left 2 bits to setup for the Q clocks
	rlcf	PDCbuf_L			;  .
	rlcf	PDCbuf_H			;  .
	bcf		STATUS,C			;  .
	rlcf	PDCbuf_L			;  .
	rlcf	PDCbuf_H			;shift PDC left 2 bits to setup for the Q clocks			;  .
	movf	PDCbuf_H,w
	btfsc	TestModeActive_b
		movlw 0x01				;this forces a ~25% PWM duty cycle for test mode 
	movwf	PDC0H
	movwf	PDC1H
	movwf	PDC2H	
	movwf	PDC3H	
	movf	PDCbuf_L,w
	btfsc	TestModeActive_b
		movlw 0xF3				;this forces a ~25% PWM duty cycle for test mode 
	movwf	PDC0L	
	movwf	PDC1L
	movwf	PDC2L
	movwf	PDC3L
	bcf		PWMCON1,UDIS		;Enable the PWM buffer update
xLOAD_PDCS
	return
	
INIT_PERIPHERALS 
;******************************************************************************
; ADC initialization
; ADC is initialized to read he Potentiometer connected to AN1
; Motor current is connected to AN0
;******************************************************************************

;..............................................................................
; Initialize the ADC
;..............................................................................
	movlw	b'00010001'	;Multi channel, Sequential mode conversion of Group A&B
	movwf	ADCON0		;  using group A (AN0) and group B (AN1), single shot. 
	movlw	b'00010000'	;FIFO enabled, Vref+=AVdd, Vref-= AVss
	movwf	ADCON1
	movlw	b'11110010'	;Right justified result, 48Tad, Fosc/32 clock
	movwf	ADCON2
	movlw	b'01000000'	;Int every 2nd and 4th words written to FIFO. 
	movwf	ADCON3
	movlw	b'00000000'	;AN0 and AN1 (PA0 and PA1) selected for groups A and B
	movwf	ADCHS
	movlw	b'00000011'	;AN0 and AN1 selected as analog inputs
	movwf	ANSEL0
	movlw	b'00000000'	
	movwf	ANSEL1
;..............................................................................
;Initializing the motion feedback module to read Hall sensors using Input Capture 
;HallA/B/C @ IC1/IC2/IC3 with Timer5 turned on with 1:8 prescale
;..............................................................................
	bsf		TRISA,2			;IC1/2/3 inputs
	bsf		TRISA,3
	bsf		TRISA,4
	movlw	b'00011001'		;Timer5 ON with prescale 1:8
	movwf	T5CON
	movlw	b'01001000'		;Cap1/2/3-capture every input state change
	movwf	CAP1CON
	movlw	b'01001000'		;Cap1/2/3-capture every input state change
	movwf	CAP2CON
	movlw	b'01001000'		;Cap1/2/3-capture every input state change
	movwf	CAP3CON
	movlw	b'00000000'		;Digital filters disabled
	movwf	DFLTCON
	movlw	b'00000000'		;Disable QEI
	movwf	QEICON
;..............................................................................
;PWM timer is in free running mode(Edge aligned)
; with Fosc/4 input clock 
;..............................................................................
	movlw	b'00000000'
	movwf	PTCON0
;.............................................................................
; Setup PWM Freq
; for Fosc = 40Mz, PTPER= 0x1F3 for 20kHZ pwm frequency
;..............................................................................
	movlw	LOW(PTPER_VALUE)				
	movwf	PTPERL								
	movlw	HIGH(PTPER_VALUE)
	movwf	PTPERH
;..............................................................................
; PWM<0:5> PWM Duty cycle on overide 
;..............................................................................
	movlw	b'00000000'			
	movwf	OVDCOND
;..............................................................................
; All PWMs = 0 on init
;..............................................................................
	movlw	b'00000000'			
	movwf	OVDCONS
;..............................................................................
; PCPWM outputs initialized in independent mode.
; PWM0..PWM5 are active outputs. All bridge FETs are off
; are off.
;..............................................................................
	movlw	b'01000111'			
	movwf	PWMCON0	
;..............................................................................
; Output overides are asynchronouse with the PWM timebase. 
;..............................................................................
	movlw	b'00000000'			
	movwf	PWMCON1
;..............................................................................
; setup dead time for 1.0 usec (FOSC/2 prescaler at 40 MHz)
;..............................................................................
	movlw	b'01010100'			
	movwf	DTCON				
;.............................................................................
; restore to b'10000011' ;FaultA current limit in cycle-by-cycle mode
; ;..............................................................................
	movlw	b'00000011'			
	movwf	FLTCONFIG			
;..............................................................................
; Special event not used
;..............................................................................
	movlw	0x00				
	movwf	SEVTCMPL
	movlw	0x00
	movwf	SEVTCMPH
;..............................................................................
; Initialize all PWM duty cycle registers with the minimum duty cycle value
;..............................................................................
	movlw	low(MinPWM)
	movwf	PDCbuf_L
	movlw	high(MinPWM)
	movwf	PDCbuf_H	
	call 	LOAD_PDC			
	clrf	PTCON1					;PWM timer OFF with time base counting up.
;..............................................................................
; Initialize Timer1 for 6.554 msec interrupt. This interrupt, along with the
;  HALL_A_INT interrupt, is used to detect that the motor is not rotating.
;  Timer1 interrupt increments a counter, HALL_A_High interrupt clears the
;  counter. When Hall_A_High interrupts stop (because the motor is not rotating),
;  the counter overflows and clears the MotorRunning_b flag. The HAll_A_HIGH
;  interrupt sets the MotorRunning_b flag. 
;..............................................................................
	movlw	b'10000001'				;Timer1 is setup for 16 bit reads,
	movwf	T1CON					;  prescale counter is 1:1,
									;  external timer1 oscillator is off,
									;  and internal clock is selected (Fosc/4)
									;  Timer1 is enabled,
	bcf		PIE1,TMR1IE				;Timer1 interrupt is not enabled
	bsf		IPR1,TMR1IP				;Timer1 interrupt is set as high priority 
;..............................................................................
; Initialize Timer0 for ~.5 sec interrupt. Used by blinking LED.
;	Timer0 is enabled with 16 bits, using internal clock source with prescaller
;	of 1:64 for an interrupt period of 0.419 seconds(1000H x 100e-9 x 64) 
;..............................................................................
	movlw	b'10000101'				
	movwf	T0CON
xINIT_PERIPHERALS	
	RETURN
	
PBUTTON_STATUS
;*******************************************************************************
;This routine monitors and debounces the switch status of the Start/Stop,
; Reverse, and Brake push-button switches. The status of the switches are set
; and cleared after the debouncing period. The button status flags are cleared
; by the routine that services the corresponding switch input. 
;
; Note that only one Pushbutton is recognized at a time with Start/Stop having
; highest priority followed by Reverse and Brake. 
;*******************************************************************************

	EnableMux PushButton, PButtonInputs
		
	movlw	PButtonMask				
	andwf	PButtonInputs,w			;isolate the switch inputs
	cpfseq	PrevSwInputs		
		bra	SwitchChanged
NoSwitchChange
	movf	DebounceCntr
	bz		xPBUTTON_STATUS			
	decfsz	DebounceCntr
		bra xPBUTTON_STATUS		;switch change is not yet stable
		
	bcf 	StartStop_b			;clear all pbutton flags
	bcf 	Reverse_b
	bcf 	Brake_b
	btfsc	WREG,StartStop
		goto SetStartStop_b
	btfsc	WREG,Reverse
		goto SetReverse_b
	btfsc	WREG,Brake
		goto SetBrake_b	
	bra		xPBUTTON_STATUS
SetStartStop_b
	bsf		StartStop_b
	btg		StopMotor_b	
	bra		xPBUTTON_STATUS
SetReverse_b
	bsf		Reverse_b
	bsf		ReverseEnabled_b
	btg		ForwardMotor_b
	bra		xPBUTTON_STATUS
SetBrake_b	
	bsf		Brake_b
	bra		xPBUTTON_STATUS
SwitchChanged
	movwf	PrevSwInputs	
	movlw	DebounceCnt			;restart counter
	movwf	DebounceCntr	
xPBUTTON_STATUS
	return	

PROCESS_PBUTTONS
;*******************************************************************************
; This routine services the push button switches 
;*******************************************************************************
	btfsc	StartStop_b			
		goto StartStopMotor
	btfsc	Reverse_b
		goto ReverseMotor
	btfsc	Brake_b					;This switch is for fast braking
		goto BrakeMotor		
	;TurnOffLED Reverse
	bra	xPROCESS_PBUTTONS
;...................................
; Fast Brake the Motor
;...................................
BrakeMotor
	call	BRAKE_MOTOR
	bra		xPROCESS_PBUTTONS
;...................................
; start or stop the motor
;...................................
StartStopMotor
	btfsc	StopMotor_b			;start or stop the motor?
	   goto StopMotor
StartMotor
	TurnOffLED Brake
	TurnOffLED Ilimit
	bcf		Brake_b
	call	START_MOTOR			;restart the motor if previously stopped.
	bra		xPROCESS_PBUTTONS
StopMotor
	call	STOP_MOTOR			;STOP motor
	bra		xPROCESS_PBUTTONS
;...............................................................................
;Reverse the motor rotation from the previous rotation direction.
;  Turn on the reverse LED only when running in reverse.
;  If motor is not running, turn on the rev LED but do not stop and 
;	reverse start the motor.
;...............................................................................
ReverseMotor
	TurnOffLED Reverse
	btfsc	ForwardMotor_b
		bra	MotorStop
	TurnOnLED Reverse, NoBlink
MotorStop
	btfss	MotorRunning_b
		bra xPROCESS_PBUTTONS
	call	STOP_MOTOR				;Stop the motor before reversing the direction
	call	START_MOTOR				;restart the motor in the opposite direction
	bcf		ReverseEnabled_b
	bcf		Reverse_b
xPROCESS_PBUTTONS
	return

BRAKE_MOTOR
;*******************************************************************************
; This stops the motor as quickly as possiby
;  by setting each bridge output to low. 
;*******************************************************************************
	TurnOnLED Brake,NoBlink
	TurnOffLED Run
	bcf		SeqEnabled_b			;do not update sequence if motor is stopped
	bsf		StopMotor_b
	movlw	b'00100101'		 		;STOP motor, each 1/2 bridge to low,
	movwf	OVDCOND					; for a braking stop 
	movlw	b'01000000'
	movwf	PWMCON0		 			;PWM output pairs are in commplimentary mode
	bcf		PWMenable_b
	bcf		Brake_b
xBRAKE_MOTOR
	return

STOP_MOTOR
;*******************************************************************************
; This routine stops the motor by driving the PWMs to 0% duty cycle. The
; outputs are set to independent mode to force all bridge FETs off.
;*******************************************************************************
	TurnOffLED	Run
	TurnOffLED  Brake
	bcf		SeqEnabled_b		;do not update sequence if motor is stopped
	clrf	OVDCONS				;turn off all bridge FETs for freewheeling stop 
	movlw	b'00000000'
	movwf	OVDCOND		 		;disable PWM to outputs				
	movlw	b'01000111'
	movwf	PWMCON0		 		;PWM output pairs are in independant mode
	bcf		PWMenable_b
xSTOP_MOTOR
	bcf		StartStop_b
WaitForStop
	btfsc	MotorRunning_b
		goto WaitForStop		;MotorRunning_b is cleared by Timer0 int 
	return
	
START_MOTOR 
;*******************************************************************************
; This starts the motor.
;*******************************************************************************		
;...............................................................................
;Reinitialize FaultA in cycle by cycle mode 
;...............................................................................
	btfsc	TestModeActive_b
		bra MotorStart
	TurnOnLED Run,Blink				;don't turn on Run LED during the test mode				
MotorStart
	bcf		StopMotor_b
	movlw	b'00000011'				
	movwf	FLTCONFIG			
	bcf		OverCurFault_b			;clear over current fault flag
;...............................................................................
; When starting the Motor, get the last updated switching sequence from the 
; UPDATE_SEQUENCE ISR (called from the HallA, HallB, or HallC interrupts).
;...............................................................................
	bcf		GlobalIntEnable_b
	movff	OVDCONDbuf,OVDCOND		;These new values are recognized at the end
	movff	PWMCON0buf,PWMCON0		; of the currently active PWM period. 
	bsf		GlobalIntEnable_b
	movlw	UpdateDelayCnt
	movwf	UpdateDelayCntr
	
InitPWMdutyCycle
	movlw	low(MinPWM)
	movwf	PWMdutyCycle_L
	movlw	high(MinPWM)
	movwf	PWMdutyCycle_H			;Start motor with minimum duty cycle
	movff	PWMdutyCycle_L,PDCbuf_L	;Update all PDC registers with duty cycle
	movff	PWMdutyCycle_H,PDCbuf_H
	call	LOAD_PDC
	clrf	PTMRL
	clrf	PTMRH
	bsf		PWMenable_b
	bsf		SeqEnabled_b
xSTART_MOTOR 
	bcf		StartStop_b
	return
	
CLEAR_RAM	
;*******************************************************************************
; Initialize all ram with 0x00
;*******************************************************************************
	lfsr 	FSR0, 0x2FF			;Top of RAM space
NEXT 
	clrf	POSTDEC0
	btfss	FSR0H, 3 
			bra NEXT 	
xCLEAR_RAM	
	return	
	
	
RD_DIP_SWITCH
;******************************************************************************
; The dip switch must be setup for the desired bldc switching sequence prior to
; applying power or pushing the reset button. If an invalid dip switch value is
; selected, the LEDS will flash indicating a selection error and the program
; will stop until a correct setting is selected.
;
; Up to 8 different switching sequences are supported (dip switch bits b0..b2).
;	on exit: TBLPTRLsave = address for sequence table as selected by the
;			 dip switch setting
;******************************************************************************
	movlw	UPPER SequenceData 	;Note that all sequence data is located in
	movwf	TBLPTRU				;  high memory with TBLPTRU = 1 and TBLPTRH = F.
	movlw	HIGH SequenceData 	
	movwf 	TBLPTRH
	movlw	low SequenceData 
	movwf	TBLPTRLsave
	clrf	DipTest
	
	EnableMux DipSwitch, DipSwInputs	
	
	swapf	DipSwInputs			;put dip settings in b0..b3
	movlw	0x0F
	andwf	DipSwInputs			;clear b4..b7
	movlw	0x0F
GetSequenceAddr
	movf	DipTest,W
	cpfsgt	DipSwInputs
		bra xRD_DIP_SWITCH
	movlw	0x20
	addwf	TBLPTRLsave
	incf	DipTest
	movlw	Low EndSequenceData 
	cpfseq	TBLPTRLsave
		bra GetSequenceAddr

InvalidSetting
	movlw	0xF0			
	TurnOnLED Wdata,Blink		;turn on all LEDs
	bra $						;Reset push-button must be pressed after a valid
								; sequence number is setup in the dip switch
xRD_DIP_SWITCH
	RETURN
	
DELAY_
;*******************************************************************************
; delay loop. for w = d'100', delay is approximately 2 seconds. 
;*******************************************************************************
	movwf	DelayCntr_H
	clrf	DelayCntr_L
	clrf	WREG
	
DelayLoop
	decfsz	WREG
		bra	DelayLoop
	nop								;don't remove, used to increase delay
	nop								;don't remove, used to increase delay
	decfsz	DelayCntr_L
		bra	DelayLoop
	decfsz	DelayCntr_H
		bra	DelayLoop
	return
	
UPDATE_SEQUENCE 
;******************************************************************************
;CAUTION:
;	THIS SUBROUTINE IS USED BY THE ISR_HIGH INTERRUPT AND BY THE MainProgram. 
;   IT IS NECESSARY TO PREVENT THE ISR_HIGH FROM ACTIVATING WHILE THIS ROUTINE
;	IS ACTIVE IN THE MainLoop.
; 
; Update the winding energizing sequence based on current Hall signals.
; Note that the sequence table in initialized at start-up after selecting the
; appropriate sequence for the motor being used (up to 14 choices).
;
; The PWMCON0 bits that control independant or complimentary mode are set to
; complimentary for bit pairs that are active (10 or 01) in the OVDCOND
; register. For bit pairs that are 00, it is necessary to reconfigure that bit
; pair for independant mode so that both bridge FETs can be turned off. 
; See the SequenceData table for more information.
;
;******************************************************************************

;	movf	PORTA,W
;	andlw	b'00011100'	
;	movwf	TBLPTRL	
;	movlw	b'00010100'							
;	CPFSEQ	TBLPTRL
;		bra start
;	nop
;	bra start

	movff	TBLPTRLsave,TBLPTRL
	movf	PORTA,W					;Load porta value to WREG
	andlw	b'00011100'				;IC1/IC2/IC3 are on PORTA pin 2,3,4					
	rrncf	WREG,W					;shift bits right to address the table with offset
	addwf	TBLPTRL
	btfsc	ForwardMotor_b			;Check the direction command
		bra	GetPWMSequence
AddRevOffset	
	movlw	0x10					;This offset is for reading the reverse
	addwf	TBLPTRL					; sequence data					 
GetPWMSequence
	TBLRD*+
	movff	TABLAT,PWMCON0buf		;Pick the PWMCON0 value from table
	TBLRD* 
	movff	TABLAT,OVDCONDbuf		;Pick the OVDCOND value from table 
	btfss	SeqEnabled_b			;update only if motor is running
		bra xUPDATE_SEQUENCE
	bcf		PWMenable_b
	movff	PWMCON0buf,PWMCON0
	movff	OVDCONDbuf,OVDCOND	
	bsf		PWMenable_b
	
xUPDATE_SEQUENCE	
	return	
							
;**************************************************************************
; TestMode is found in the following file
;**************************************************************************
	include		"HIP4086 TestMode.inc"

;**************************************************************************
; Bit definitions in sequence table for MA, MB, and MC bit pairs of the
; OVDCOND register:
;
;  10	PWM is active  
;  01	Low side bridge FET is active (on)
;  00	Both bridge FETs off.
; 
;  The two left most bits of OVDCOND are aways 00
;
; Bit definitions of the PWMCON0 register:
;
;	b7 is always 0.
;	b6 is always 1 (pwm enable)
;	b5,4,3 are always 0
;	b2,1,0 are set to select the one Mx pair as Z by configuring
;		the corresponding output pair as independant. The other two pairs are
;		configured as complementary. 
;
;	example:
;		   PWMCON0		 OVDCOND
;						 00MCMBMA
;		b'01000001',db b'00011000'; LPZ
;
;		for OVDCOND register
;			MA bits (00) = off (Z)
;			MB bits (10) = PWM (P)
;			MC bits (01) = Low (L)
;
; 	    for PWMCON0 register:
; 		b0 is set to configure the MA pair as independent
; 		b1 is reset to configure the MB pair as complementary
; 		b2 is reset to configure the MC pair as complementary
;
;**************************************************************************
; The following definitions specify the bit patterns as explained above
;			  PWMCON0_OVDCOND_
#define ZLP	b'0000011001000100'
#define ZPL b'0000100101000100'	
#define LZP	b'0001001001000010'
#define LPZ b'0001100001000001'
#define PLZ b'0010010001000001'	
#define PZL b'0010000101000010'	
#define ZZZ b'0000000001000111'
; The following switching sequence codes are used only by the TEST_MODE routine
#define LLL b'0001010101000000' 
#define LLP b'0001011001000000'
#define	LPL b'0001100101000000'
#define PLL b'0010010101000000'
;**************************************************************************

;The following data decodes all possible combinations of the Hall inputs.
;Note that the DSWxxx label refers to the corresponding Dip Switch setting.
;The REVxxxx label refers to the reverse rotation sequence corresponding
;to the same dip switch setting as DSWxxxx.
					code 0x1E00
SequenceData
;Hall addr:	000 001 010 011 100 101 110 111
DSW0000 dw 	ZPL,PZL,ZZZ,PLZ,LPZ,ZZZ,LZP,ZLP
REV0000	dw 	ZLP,LZP,ZZZ,LPZ,PLZ,ZZZ,PZL,ZPL

DSW0O01	dw 	PZL,ZPL,PLZ,ZZZ,ZZZ,LPZ,ZLP,LZP
REV0001	dw 	LZP,ZLP,LPZ,ZZZ,ZZZ,PLZ,ZPL,PZL

DSW0010	dw 	ZZZ,PLZ,ZPL,PZL,LZP,ZLP,LPZ,ZZZ 
REV0010	dw 	ZZZ,LPZ,ZLP,LZP,PZL,ZPL,PLZ,ZZZ

DSW0011	dw 	PLZ,ZZZ,PZL,ZPL,ZLP,LZP,ZZZ,LPZ 
REV0011	dw 	LPZ,ZZZ,LZP,ZLP,ZPL,PZL,ZZZ,PLZ

DSW0100	dw 	LPZ,ZZZ,LZP,ZLP,ZPL,PZL,ZZZ,PLZ
REV0100	dw 	PLZ,ZZZ,PZL,ZPL,ZLP,LZP,ZZZ,LPZ 

DSW0101	dw 	ZZZ,LPZ,ZLP,LZP,PZL,ZPL,PLZ,ZZZ ;B&D proto motor
REV0101	dw 	ZZZ,PLZ,ZPL,PZL,LZP,ZLP,LPZ,ZZZ 

DSW0110	dw 	LZP,ZLP,LPZ,ZZZ,ZZZ,PLZ,ZPL,PZL
REV0110	dw 	PZL,ZPL,PLZ,ZZZ,ZZZ,LPZ,ZLP,LZP

DSW0111	dw 	ZLP,LZP,ZZZ,LPZ,PLZ,ZZZ,PZL,ZPL ;Ametek 119056, Bodine 3304
REV0111 dw 	ZPL,PZL,ZZZ,PLZ,LPZ,ZZZ,LZP,ZLP
EndSequenceData

TestSequence	
		dw 	LLL,LLL,LLL,PLL,LLL,LPL,LLP,LLL ;this sequence is used only by
											; the TEST_MODE routine
EndTestSequence

;*******************************************************************************
;   THE
	END
;*******************************************************************************
;...............................................................................
; For convenient reference, the following status register definitions are copied
; directly from the PIC18F2431 data sheet.
;...............................................................................


PTCON0: PWM TIMER CONTROL REGISTER 0
	bit 7-4 PTOPS3:PTOPS0: PWM Time Base Output Postscale Select bits
		0000 =1:1 Postscale
		0001 =1:2 Postscale
		....
		1111 =1:16 Postscale
	bit 3-2 PTCKPS1:PTCKPS0: PWM Time Base Input Clock Prescale Select bits
		00 =PWM time base input clock is Fosc/4 (1:1 prescale)
		01 =PWM time base input clock is Fosc/16 (1:4 prescale)
		10 =PWM time base input clock is Fosc/64 (1:16 prescale)
		11 =PWM time base input clock is Fosc/256 (1:64 prescale)
	bit 1-0 PTMOD1:PTMOD0: PWM Time Base Mode Select bits
		11 =PWM time base operates in a Continuous Up/Down mode for double ints.
		10 =PWM time base operates in a Continuous Up/Down Counting mode.
		01 =PWM time base configured for Single-shot mode.
		00 =PWM time base operates in a Free Running mode.
;...............................................................................
PTCON1: PWM TIMER CONTROL REGISTER 1
	bit 7 	PTEN: PWM Time Base Timer Enable bit
		1 = PWM time base is ON
		0 = PWM time base is OFF
	bit 6 PTDIR: PWM Time Base Count Direction Status bit
		1 = PWM time base counts down.
		0 = PWM time base counts up.
	bit 5-0 Unimplemented: Read as 0.
;...............................................................................
PWMCON0: PWM CONTROL REGISTER 0	
	bit 7 Unimplemented: Read as 0.
	bit 6-4 PWMEN2:PWMEN0: PWM Module Enable bits(1)
		111 =All odd PWM I/O pins enabled for PWM output(2).
		110 =PWM1, PWM3 pins enabled for PWM output.
		101 =All PWM I/O pins enabled for PWM output(2) .
		100 =PWM0, PWM1, PWM2, PWM3, PWM4 and PWM5 pins enabled for PWM output.
		011 =PWM0, PWM1, PWM2 and PWM3 I/O pins enabled for PWM output.
		010 =PWM0 and PWM1 pins enabled for PWM output.
		001 =PWM1 pin is enabled for PWM output.
		000 =PWM module disabled. All PWM I/O pins are general purpose I/O.
	bit 3-0 PMOD3:PMOD0: PWM Output Pair Mode bits
		For PMOD0:
		1 = PWM I/O pin pair (PWM0, PWM1) is in the Independent mode.
		0 = PWM I/O pin pair (PWM0, PWM1) is in the Complementary mode.
		For PMOD1:
		1 = PWM I/O pin pair (PWM2, PWM3) is in the Independent mode.
		0 = PWM I/O pin pair (PWM2, PWM3) is in the Complementary mode.
		For PMOD2:
		1 = PWM I/O pin pair (PWM4, PWM5) is in the Independent mode.
		0 = PWM I/O pin pair (PWM4, PWM5) is in the Complementary mode.
		For PMOD3(3):
		1 = PWM I/O pin pair (PWM6, PWM7) is in the Independent mode.
		0 = PWM I/O pin pair (PWM6, PWM7) is in the Complementary mode.	
;...............................................................................
PWMCON1: PWM CONTROL REGISTER 1
	bit 7-4 SEVOPS3:SEVOPS0: PWM Special Event Trigger Output Postscale 
			Select bits
		0000 =1:1 Postscale
		0001 =1:2 Postscale
		....
		1111 =1:16 Postscale
	bit 3 SEVTDIR: Special Event Trigger Time Base Direction bit
		1 = A special event trigger will occur when the PWM time base is 
			counting downwards.
		0 = A special event trigger will occur when the PWM time base is 
			counting upwards.
	bit 2 Unimplemented: Read as 0.
	bit 1 UDIS: PWM Update Disable bit
		1 = Updates from duty cycle and period buffer registers are disabled.
		0 = Updates from duty cycle and period buffer registers are enabled.
	bit 0 OSYNC: PWM Output Override Synchronization bit
		1 = Output overrides via the OVDCON register are synchronized to the 
			PWM time base.
		0 = Output overrides via the OVDCON register are asynchronous.
;...............................................................................
OVDCOND: OUTPUT OVERRIDE CONTROL REGISTER
--------------------------------------------------------------
R/W-1	|R/W-1	|R/W-1	|R/W-1	|R/W-1	|R/W-1	|R/W-1	|R/W-1
--------------------------------------------------------------
POVD7(1)|POVD6(1)|POVD5	|POVD4	|POVD3	|POVD2	|POVD1	|POVD0
--------------------------------------------------------------
bit 7	|												|bit 0
--------------------------------------------------------------

bit 7-0 POVD7:POVD0: PWM Output Override bits(1)
		1 = Output on PWM I/O pin is controlled by the value in the Duty Cycle
			register and the PWMtime base.
		0 = Output on PWM I/O pin is controlled by the value in the
			corresponding POUT bit.
		Note 1: Bits 7-6 unimplemented in PIC18F2X31 devices. Maintain these 
				bits clear.
;...............................................................................
OVDCONS: OUTPUT STATE REGISTER
R/W-0	R/W-0	R/W-0	R/W-0	R/W-0	R/W-0	R/W-0	R/W-0
POUT7	POUT6	POUT5	POUT4	POUT3	POUT2	POUT1	POUT0
bit 7 													bit 0

bit 7-0 POUT7:POUT0: PWM Manual Output bits(1)
		1 = Output on PWM I/O pin is ACTIVE when the corresponding PWM output 
			override bit is cleared.
		0 = Output on PWM I/O pin is INACTIVE when the corresponding PWM output 
			override bit is cleared.
		Note 1: Bits 7-6 unimplemented in PIC18F2X31 devices. Maintain these
				bits clear. 
;...............................................................................
REGISTER 20-1: ADCON0: A/D CONTROL REGISTER 0
--------------------------------------------------------------
  U-0 	|  U-0 	|R/W-0	|R/W-0	|R/W-0	|R/W-0	|R/W-0	|R/W-0
--------------------------------------------------------------
   	|   	|ACONV	|ACSCH	|ACMOD1	|ACMOD0	|GO/DONE|ADON
--------------------------------------------------------------
bit 7	|												|bit 0
--------------------------------------------------------------

bit 7-6 Unimplemented: Read as 0
bit 5 ACONV: Auto-Conversion Continuous Loop or Single-shot Mode Select bit
	1 = Continuous Loop mode Enabled
	0 = Single-shot mode Enabled
bit 4 ACSCH: Auto-Conversion Single or Multi-Channel mode bit
	1 = Multi-Channel mode Enabled, Single Channel mode Disabled
	0 = Single Channel mode Enabled, Multi-Channel mode Disabled
bit 3-2 ACMOD: Auto-Conversion mode Sequence Select bits
If ACSCH = 1:
	00 =Sequential Mode1 (SEQM1). Two samples are taken in sequence:
	1st sample: Group A
	2nd sample: Group B
	01 =Sequential Mode2 (SEQM2). Four samples are taken in sequence:
	1st sample: Group A
	2nd sample: Group B
	3rd sample: Group C
	4th sample: Group D
	10 =Simultaneous Mode1 (STNM1). Two samples are taken simultaneously:
	1st sample: Group A and Group B
	11 =Simultaneous Mode2 (STNM2). Two samples are taken simultaneously:
	1st sample: Group A and Group B
	2nd sample: Group C and Group D
If ACSCH = 0, Auto-Conversion Single Channel Sequence mode enabled:
	00 =Single Ch Mode1 (SCM1). Group A is taken and converted
	01 =Single Ch Mode2 (SCM2). Group B is taken and converted
	10 =Single Ch Mode3 (SCM3). Group C is taken and converted
	11 =Single Ch Mode4 (SCM4). Group D is taken and converted
Note: Group A, B, C, D refer to the ADCHS register.
bit 1 GO/DONE: A/D Conversion Status bit
	1 = A/D conversion cycle in progress. 
		Setting this bit starts the A/D	conversion cycle. If Auto-Conversion
		Single-shot mode is enabled (ACONV = 0),	this bit is automatically
		cleared by hardware when the A/D conversion (single	or multi-channel 
		depending on ACMOD settings) has completed. If Auto- Conversion 
		Continuous Loop mode is enabled (ACONV = 1), this bit remains set after 
		the	user/trigger has set it	(continuous conversions). It may be	cleared	
		manually by the user to stop the conversions.
	0 = A/D conversion or multiple conversions completed/not in progress
bit 0 ADON: A/D On bit
	1 = A/D converter module is enabled (after brief power-up delay, 
		starts continuous sampling)
	0 = A/D converter module is disabled
;...............................................................................
REGISTER 20-2: ADCON1: A/D CONTROL REGISTER 1
---------------------------------------------------------------
R/W-0	|R/W-0	|U-0	|R/W-0	|R-0	|R-0	|R-0	|R-0
---------------------------------------------------------------
VCFG	|VCFG	|  	 	|FIFOEN	|BFEMT	|BFOVFL	|ADPNT1	|ADPNT0
---------------------------------------------------------------
bit 7	|												|bit 0
---------------------------------------------------------------

bit 7-6 VCFG<1:0>: A/D VREF+ and A/D VREF- Source Selection bits
	00 =VREF+ = AVDD, VREF- = AVSS, 
		(AN2 and AN3 are Analog inputs or Digital I/O)
	01 =VREF+ = External VREF+, VREF- = AVSS, 
		(AN2 is an Analog input or Digital I/O)
	10 =VREF+ = AVDD, VREF- = External VREF-, 
		(AN3 is an Analog input or Digital I/O)
	11 =VREF+ = External VREF-, VREF- = External VREF
bit	5 Unimplemented: Read as 0
bit 4 FIFOEN: FIFO Buffer Enable bit
	1 = FIFO is enabled
	0 = FIFO is disabled
bit 3 BFEMT: Buffer Empty bit
	1 = FIFO is empty
	0 = FIFO is not empty 
		(at least one of four locations has unread A/D result data)
bit 2 BFOVFL: Buffer Overflow bit
	1 = A/D result has overwritten a buffer location that has unread data
	0 = A/D result has not overflowed
bit 1-0 ADPNT<1:0>: Buffer Read Pointer Locations bits
		Designates the location to be read next.
	00 = Buffer address 0
	01 = Buffer address 1
	10 = Buffer address 2
	11 = Buffer address 3
;...............................................................................
T0CON: TIMER0 CONTROL REGISTER
R/W-1	R/W-1	R/W-1	R/W-1	R/W-1	R/W-1	R/W-1	R/W-1
TMR0ON	T016BIT	T0CS	T0SE	PSA		T0PS2	T0PS1	T0PS0
bit 7 													bit 0

Legend:
R = Readable bit	W = Writable bit	U = Unimplemented bit, read as 0
-n = Value at POR	1 = Bit is set	0 = Bit is cleared x = Bit is unknown

bit 7 TMR0ON: Timer0 On/Off Control bit
	1 = Enables Timer0
	0 = Stops Timer0
bit 6 T016BIT: Timer0 16-Bit Control bit
	1 = Timer0 is configured as an 8-bit timer/counter
	0 = Timer0 is configured as a 16-bit timer/counter
bit 5 T0CS: Timer0 Clock Source Select bit
	1 = Transition on T0CKI pin input edge
	0 = Internal clock (FOSC/4)
bit 4 T0SE: Timer0 Source Edge Select bit
	1 = Increment on high-to-low transition on T0CKI pin
	0 = Increment on low-to-high transition on T0CKI pin
bit 3 PSA: Timer0 Prescaler Assignment bit
	1 = TImer0 prescaler is not assigned. Timer0 clock input bypasses prescaler.
	0 = Timer0 prescaler is assigned. Timer0 clock input comes from prescaler output.
bit 2-0 T0PS<2:0>: Timer0 Prescaler Select bits
	111 = 1:256 Prescale value
	110 = 1:128 Prescale value
	101 = 1:64 Prescale value
	100 = 1:32 Prescale value
	011 = 1:16 Prescale value
	010 = 1:8 Prescale value
	001 = 1:4 Prescale value
	000 = 1:2 Prescale value
;...............................................................................
;...............................................................................
;...............................................................................
;...............................................................................
;...............................................................................
;...............................................................................
;...............................................................................
;...............................................................................
;SequenceData0 		; for B&D PROTO Motor
;; Forward (CW) 
;; 		   PWMCON0	  OVDCOND	   BRIDGE    HALL	
;;		  			  PDMCMBMA	   OUTPUT	C B	A
;	db	b'01000111',b'00000000' ;	ZZZ		0 0	0		<- error condition
;	db	b'01000100',b'00000110' ;	ZLP		0 0	1
;	db	b'01000001',b'00011000' ;	LPZ		0 1	0
;	db	b'01000010',b'00010010' ;	LZP		0 1	1		P = BRIDGE OUTPUT PWM			
;	db	b'01000010',b'00100001' ;	PZL		1 0	0		Z = BRIDGE OUTPUT OFF	
;	db	b'01000001',b'00100100' ;	PLZ		1 0	1		L = BRIDGE OUTPUT LOW
;	db	b'01000100',b'00001001' ;	ZPL		1 1	0
;	db	b'01000111',b'00000000' ;	ZZZ		1 1	1		<- error condition
;RevSeqData0 ;(CCW)
;	db	b'01000111',b'00000000' ;	ZZZ		0 0 0	
;	db	b'01000001',b'00011000' ;	LPZ		0 0 1
;	db	b'01000010',b'00100001' ;	PZL		0 1 0
;	db	b'01000100',b'00001001' ;	ZPL		0 1 1
;	db	b'01000100',b'00000110' ;	ZLP		1 0 0
;	db	b'01000010',b'00010010' ;	LZP		1 0 1
;	db	b'01000001',b'00100100' ;	PLZ		1 1 0
;	db	b'01000111',b'00000000' ;	ZZZ		1 1 1
;;...............................................................................	
;SequenceData1  		;Ametec119056 (for Polo Zou)
;; 		   PWMCON0	  OVDCOND	   BRIDGE    HALL	
;;		  			  xxMCMBMA	   OUTPUT	C B	A		
;	db	b'01000100',b'00000110' ;	ZLP		0 0 0
;	db	b'01000010',b'00010010' ;	LZP		0 0 1
;	db	b'01000111',b'00000000' ;	ZZZ		0 1 0
;	db	b'01000001',b'00011000' ;	LPZ		0 1 1
;	db	b'01000001',b'00100100' ;	PLZ		1 0 0
;	db	b'01000111',b'00000000' ;	ZZZ		1 0 1
;	db	b'01000010',b'00100001' ;	PZL		1 1 0
;	db	b'01000100',b'00001001' ;	ZPL		1 1 1
;RevSeqData1 ;(CCW)
;	db	b'01000100',b'00001001' ;	ZPL		0 0 0
;	db	b'01000010',b'00100001' ;	PZL		0 0 1
;	db	b'01000111',b'00000000' ;	ZZZ		0 1 0
;	db	b'01000001',b'00100100' ;	PLZ		0 1 1
;	db	b'01000001',b'00011000' ;	LPZ		1 0 0
;	db	b'01000111',b'00000000' ;	ZZZ		1 0 1
;	db	b'01000010',b'00010010' ;	LZP		1 1 0
;	db	b'01000100',b'00000110' ;	ZLP		1 1 1
;;...............................................................................	
;SequenceData2  		;motor type used in PICDEM MCLV
;; 		   PWMCON0	  OVDCOND	   BRIDGE    HALL	
;;		  			  xxMCMBMA	   OUTPUT	C B	A		
;	db	b'01000111',b'00000000'	;	ZZZ		0 0 0
;	db	b'01000010',b'10010010'	;	LZP		0 0 1
;	db	b'01000001',b'01100100'	;	PLZ		0 1 0
;	db	b'01000100',b'00000110' ;	ZLP		0 1 1
;	db	b'01000100',b'00001001' ;	ZPL		1 0 0
;	db	b'01000001',b'01011000' ;	LPZ		1 0 1
;	db	b'01000010',b'10100001' ;	PZL		1 1 0
;	db	b'01000111',b'00000000' ;	ZZZ		1 1 1
;RevSeqData2 ;(CCW)
;	db	b'01000111',b'00000000'	;	ZZZ		0 0 0
;	db	b'01000001',b'00010010'	;	LPZ		0 0 1
;	db	b'01000001',b'00100100'	;	PLZ		0 1 0
;	db	b'01000100',b'00000110' ;	ZLP		0 1 1
;	db	b'01000100',b'00001001' ;	ZPL		1 0 0
;	db	b'01000001',b'00011000' ;	LPZ		1 0 1
;	db	b'01000010',b'00100001' ;	PZL		1 1 0
;	db	b'01000111',b'00000000' ;	ZZZ		1 1 1
;;...............................................................................	
